## Geophysical Research Letters
## Andrade et al. (2025)
## Supporting Information for:
## "Controls on Water-Magma Interactions at Hydraulically-charged Volcanic Islands"


library(gridExtra)
library(scales)
library(ggplot2)
library(reshape2)
library(dplyr)
library(tidyr)
library (cowplot)


# Force the locale to English for date labels
Sys.setlocale("LC_TIME", "C")

# Read the datasets
regression <- read.csv2("Dataset/Regression.csv", header = TRUE)
Azores <- read.csv2("Dataset/Azores.csv", header = TRUE)

# Convert the 'Date' column to date type
#Azores$Date <- as.Date(Azores$Date, format = "%d/%m/%Y")

# List of stations in the desired order (by elevation)
station_order <- c("Ponta.Delgada", 
                   "Fazenda.de.Santa.Cruz", 
                   "Terreiros", 
                   "Pico.da.Casinha", 
                   "Lagoa.da.Caldeira.Rasa")

# Transform the data to long format
Azores_long <- Azores %>%
  pivot_longer(cols = all_of(station_order), names_to = "Station", values_to = "Precipitation") %>%
  arrange(factor(Station, levels = station_order), Date)

# Calculate cumulative precipitation per station
Azores_long <- Azores_long %>%
  group_by(Station) %>%
  mutate(Cumulative_Precipitation = cumsum(Precipitation)) %>%
  ungroup()

# Create a color palette based on the order of the stations
colors <- c("#fedc97", "#b5b682", "#7c9885", "#28666e", "#033f63")
names(colors) <- station_order  # Assign names to the colors

# First plot: stacked bars and cumulative precipitation lines
plot1 <- ggplot(Azores_long, aes(x = Date)) +
  geom_bar(aes(y = Precipitation, fill = factor(Station, levels = station_order)), alpha = .8,
           stat = "identity", position = position_stack(reverse = TRUE)) +  # Reverse stacking
  geom_line(aes(y = Cumulative_Precipitation, color = factor(Station, levels = station_order), group = Station), 
            size = 1) +
  scale_fill_manual(values = colors) +  # Assign colors correctly
  scale_color_manual(values = colors) +
  scale_y_continuous(limits = c(0, 3500), breaks = seq(0, 3500, by = 500)) +  # Y-axis from 0 to 3500 with 500 intervals
  scale_x_discrete()
labs(title = "Monthly and Cumulative Precipitation per Station (Stacked - Inverted Order)",
     x = "Date",
     y = "Precipitation (mm)") +
  theme_classic() +
  theme(axis.text.x = element_text(angle = 45, hjust = 1),
        legend.position = "none")  # Remove legend

# Second plot: regression plot with elevation vs precipitation
plot2 <- ggplot(regression, aes(x = Elevation, y = pp..mm.yr., 
                                color = factor(Station, levels = station_order))) +
  geom_point(size = 3) +   # Add station points
  geom_smooth(method = "lm", se = FALSE, color = "black", linetype = "dashed", size = 0.5) +  # Add linear regression line
  geom_smooth(method = "lm", formula = y ~ poly(x, 2), se = FALSE, color = "black", linetype = "dashed", size = 0.5) +  # Polynomial regression (degree 2)
  scale_color_manual(values = colors) +  # Assign colors to each station
  geom_text(aes(label = Station), hjust = -0.1, vjust = 1.5, size = 3) +  # Add labels near each point
  scale_x_continuous(breaks = seq(0, 600, by = 100), limits = c(0, 600)) +  # Ensure 600 is included
  scale_y_continuous(limits = c(1000, 3000), breaks = seq(1000, 3000, by = 500)) +  # Y-axis from 1000 to 3500 with 500 intervals
  labs(title = "Relationship between Elevation and Annual Precipitation",
       x = "Elevation (m)",
       y = "Precipitation (mm/year)") +
  theme_classic() +
  theme(axis.text.x = element_text(angle = 0), legend.position = "none")  # Remove legend


# Create layout where the second plot occupies only half the horizontal space
layout <- rbind(c(1),
                c(2, 3))  # Space for another figure to the right of plot2

# Empty space where you can add your figure later
empty_plot <- ggplot() + theme_void()

# Arrange the plots with the desired layout
grid.arrange(plot1, plot2, empty_plot, layout_matrix = layout)